<?php

namespace App\Http\Controllers;

use App\Models\ConsumoOp;
use App\Models\Op;
use App\Models\Material;
use Illuminate\Http\Request;

class ConsumoOpController extends Controller
{
    public function index()
    {
        $consumos = ConsumoOp::with(['op', 'material'])->orderBy('created_at', 'desc')->get();
        return view('consumos.index', compact('consumos'));
    }

    public function create()
    {
        $ops = Op::all();
        $materiais = Material::all();
        return view('consumos.create', compact('ops', 'materiais'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'op_id' => 'required|exists:ops,id',
            'material_id' => 'required|exists:materiais,id',
            'quantidade' => 'required|numeric|min:0.01',
        ]);

        $material = Material::findOrFail($request->material_id);

        $consumo = ConsumoOp::create([
            'op_id' => $request->op_id,
            'material_id' => $material->id,
            'quantidade' => $request->quantidade,
            'unidade' => $material->unidade,
            'categoria' => $material->categoria,
        ]);

        // Abate automático do estoque
        $material->decrement('estoque_atual', $request->quantidade);

        return redirect()->route('consumos.index')->with('success', 'Consumo registrado e estoque atualizado.');
    }

    public function show(ConsumoOp $consumo)
    {
        return view('consumos.show', compact('consumo'));
    }

    public function edit(ConsumoOp $consumo)
    {
        $ops = Op::all();
        $materiais = Material::all();
        return view('consumos.edit', compact('consumo', 'ops', 'materiais'));
    }

    public function update(Request $request, ConsumoOp $consumo)
    {
        $request->validate([
            'op_id' => 'required|exists:ops,id',
            'material_id' => 'required|exists:materiais,id',
            'quantidade' => 'required|numeric|min:0.01',
        ]);

        // Repor o estoque antigo
        $consumo->material->increment('estoque_atual', $consumo->quantidade);

        $material = Material::findOrFail($request->material_id);

        $consumo->update([
            'op_id' => $request->op_id,
            'material_id' => $material->id,
            'quantidade' => $request->quantidade,
            'unidade' => $material->unidade,
            'categoria' => $material->categoria,
        ]);

        // Abater novo estoque
        $material->decrement('estoque_atual', $request->quantidade);

        return redirect()->route('consumos.index')->with('success', 'Consumo atualizado e estoque recalculado.');
    }

    public function destroy(ConsumoOp $consumo)
    {
        // Repor estoque ao excluir
        $consumo->material->increment('estoque_atual', $consumo->quantidade);

        $consumo->delete();

        return redirect()->route('consumos.index')->with('success', 'Consumo excluído e estoque reposto.');
    }
}
