<?php

namespace App\Http\Controllers;

use App\Models\Etapa;
use App\Models\Op;
use App\Models\Oficina;
use Illuminate\Http\Request;

class EtapaController extends Controller
{
    public function index()
    {
        $etapas = Etapa::with(['op', 'oficina'])->get();
        return view('etapas.index', compact('etapas'));
    }

    public function create()
    {
        $ops = Op::all();
        $oficinas = Oficina::all();
        return view('etapas.create', compact('ops', 'oficinas'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'op_id' => 'required|exists:ops,id',
            'oficina_id' => 'required|exists:oficinas,id',
            'etapa' => 'required|in:Compra do Tecido,Envio para São Paulo,Corte,Costura,Lavanderia',
            'status' => 'required|in:Pendente,Em Andamento,Concluido',
            'data_inicio' => 'nullable|date',
            'data_previsao' => 'nullable|date',
            'data_conclusao' => 'nullable|date',
        ]);

        $etapa = Etapa::create($request->all());

        // Criar log inicial
        \App\Models\EtapaLog::create([
            'etapa_id' => $etapa->id,
            'campo_alterado' => 'criacao',
            'valor_anterior' => null,
            'valor_novo' => 'Etapa criada',
            'usuario' => auth()->check() ? auth()->user()->name : 'Sistema',
            'data' => now(),
        ]);

        return redirect()->route('etapas.index')
            ->with('success', 'Etapa criada com sucesso.');
    }

    public function show(Etapa $etapa)
    {
        $etapa->load('logs');
        return view('etapas.show', compact('etapa'));
    }

    public function edit(Etapa $etapa)
    {
        $ops = Op::all();
        $oficinas = Oficina::all();
        return view('etapas.edit', compact('etapa', 'ops', 'oficinas'));
    }

    public function update(Request $request, Etapa $etapa)
    {
        $request->validate([
            'op_id' => 'required|exists:ops,id',
            'oficina_id' => 'required|exists:oficinas,id',
            'etapa' => 'required|in:Compra do Tecido,Envio para São Paulo,Corte,Costura,Lavanderia',
            'status' => 'required|in:Pendente,Em Andamento,Concluido',
            'data_inicio' => 'nullable|date',
            'data_previsao' => 'nullable|date',
            'data_conclusao' => 'nullable|date',
        ]);

        // Capturar valores anteriores
        $valoresAnteriores = $etapa->getOriginal();
        
        // Atualizar etapa
        $etapa->update($request->all());
        
        // Verificar mudanças e criar logs detalhados
        $camposMonitorados = [
            'status' => 'Status',
            'etapa' => 'Etapa',
            'data_inicio' => 'Data de Início',
            'data_previsao' => 'Data de Previsão',
            'data_conclusao' => 'Data de Conclusão',
            'oficina_id' => 'Oficina'
        ];

        foreach ($camposMonitorados as $campo => $label) {
            if ($valoresAnteriores[$campo] != $etapa->$campo) {
                $valorAnterior = $valoresAnteriores[$campo];
                $valorNovo = $etapa->$campo;
                
                // Formatação especial para alguns campos
                if ($campo === 'oficina_id') {
                    $oficinaAnterior = $valorAnterior ? Oficina::find($valorAnterior)->nome : 'Nenhuma';
                    $oficinaNova = $valorNovo ? Oficina::find($valorNovo)->nome : 'Nenhuma';
                    $valorAnterior = $oficinaAnterior;
                    $valorNovo = $oficinaNova;
                }
                
                if (in_array($campo, ['data_inicio', 'data_previsao', 'data_conclusao'])) {
                    $valorAnterior = $valorAnterior ? date('d/m/Y', strtotime($valorAnterior)) : 'Não definida';
                    $valorNovo = $valorNovo ? date('d/m/Y', strtotime($valorNovo)) : 'Não definida';
                }

                \App\Models\EtapaLog::create([
                    'etapa_id' => $etapa->id,
                    'campo_alterado' => $label,
                    'valor_anterior' => $valorAnterior,
                    'valor_novo' => $valorNovo,
                    'usuario' => auth()->check() ? auth()->user()->name : 'Sistema',
                    'data' => now(),
                ]);
            }
        }

        return redirect()->route('etapas.index')
            ->with('success', 'Etapa atualizada com sucesso.');
    }

    public function destroy(Etapa $etapa)
    {
        $etapa->delete();

        return redirect()->route('etapas.index')
            ->with('success', 'Etapa excluída com sucesso.');
    }
}
