<?php

namespace App\Http\Controllers;

use App\Models\Material;
use Illuminate\Http\Request;

class MaterialController extends Controller
{
    public function index()
    {
        $materiais = Material::all();
        return view('materiais.index', compact('materiais'));
    }

    public function create()
    {
        return view('materiais.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'nome' => 'required',
            'unidade' => 'required',
            'estoque_atual' => 'nullable|numeric',
            'estoque_minimo' => 'nullable|numeric',
            'categoria' => 'nullable|string',
        ]);

        $material = Material::create($request->all());

        // Se foi criado com estoque inicial, registrar movimentação
        if ($material->estoque_atual > 0) {
            \App\Models\MovimentacaoEstoque::create([
                'material_id' => $material->id,
                'tipo' => 'entrada',
                'quantidade' => $material->estoque_atual,
                'motivo' => 'Estoque inicial',
                'data' => now(),
            ]);
        }

        return redirect()->route('materiais.index')
            ->with('success', 'Material criado com sucesso.');
    }

    public function show(Material $material)
    {
        return view('materiais.show', compact('material'));
    }

    public function edit(Material $material)
    {
        return view('materiais.edit', compact('material'));
    }

    public function update(Request $request, Material $material)
    {
        $request->validate([
            'nome' => 'required',
            'unidade' => 'required',
            'estoque_atual' => 'nullable|numeric',
            'estoque_minimo' => 'nullable|numeric',
            'categoria' => 'nullable|string',
        ]);

        $estoqueAnterior = $material->estoque_atual;
        $material->update($request->all());

        // Se o estoque foi alterado, registrar movimentação
        if ($estoqueAnterior != $material->estoque_atual) {
            $diferenca = $material->estoque_atual - $estoqueAnterior;
            
            \App\Models\MovimentacaoEstoque::create([
                'material_id' => $material->id,
                'tipo' => $diferenca > 0 ? 'entrada' : 'saida',
                'quantidade' => abs($diferenca),
                'motivo' => 'Ajuste manual de estoque',
                'data' => now(),
            ]);
        }

        return redirect()->route('materiais.index')
            ->with('success', 'Material atualizado com sucesso.');
    }

    public function destroy(Material $material)
    {
        $material->delete();

        return redirect()->route('materiais.index')
            ->with('success', 'Material excluído com sucesso.');
    }
}
