<?php

namespace App\Http\Controllers;

use App\Models\Op;
use App\Models\Modelo;
use App\Models\Material;
use App\Models\ConsumoOp;
use Illuminate\Http\Request;

class OpController extends Controller
{
    public function index()
    {
        $ops = Op::with('modelo')->get();
        return view('ops.index', compact('ops'));
    }

    public function create()
    {
        $modelos = Modelo::all();
        
        // Gerar próximo número da OP automaticamente
        $proximoNumero = $this->gerarProximoNumeroOp();
        
        return view('ops.create', compact('modelos', 'proximoNumero'));
    }

    /**
     * Gera o próximo número sequencial da OP
     */
    private function gerarProximoNumeroOp()
    {
        // Buscar a última OP criada
        $ultimaOp = Op::orderBy('numero_op', 'desc')->first();
        
        if (!$ultimaOp) {
            // Se não há OPs, começar com 1
            return '1';
        }
        
        // Extrair apenas números do numero_op
        $ultimoNumero = (int) preg_replace('/[^0-9]/', '', $ultimaOp->numero_op);
        
        // Se não conseguiu extrair número, usar o ID + 1
        if ($ultimoNumero == 0) {
            $ultimoNumero = $ultimaOp->id;
        }
        
        // Retornar próximo número
        return (string) ($ultimoNumero + 1);
    }

    public function store(Request $request)
    {
        // Gerar número automaticamente se não foi fornecido
        if (empty($request->numero_op)) {
            $request->merge(['numero_op' => $this->gerarProximoNumeroOp()]);
        }

        $validator = \Validator::make($request->all(), [
            'numero_op' => 'required|unique:ops,numero_op',
            'data' => 'nullable|date',
            'descricao' => 'nullable',
            'quantidade' => 'required|integer|min:1',
            'modelo_id' => 'required|exists:modelos,id',
            'materiais' => 'array',
            'materiais.*.material_id' => 'required|exists:materiais,id',
            'materiais.*.quantidade' => 'required|numeric|min:0.01',
        ]);

        // Validação extra de estoque
        $validator->after(function ($validator) use ($request) {
            if ($request->has('materiais')) {
                foreach ($request->materiais as $linha) {
                    $material = \App\Models\Material::find($linha['material_id']);
                    if ($material && $material->estoque_atual < $linha['quantidade']) {
                        $validator->errors()->add(
                            'materiais',
                            'Estoque insuficiente para o material: ' . $material->nome . '. Disponível: ' . $material->estoque_atual
                        );
                    }
                }
            }
        });

        if ($validator->fails()) {
            // Manter os dados dos materiais na sessão para não perder
            if ($request->has('materiais')) {
                session()->flash('materiais_old', $request->materiais);
            }
            return back()->withErrors($validator)->withInput();
        }

        // Salva OP
        $op = \App\Models\Op::create([
            'numero_op' => $request->numero_op,
            'data' => $request->data,
            'descricao' => $request->descricao,
            'quantidade' => $request->quantidade,
            'modelo_id' => $request->modelo_id,
        ]);

        // Salva consumos e abate estoque
        if ($request->has('materiais')) {
            foreach ($request->materiais as $linha) {
                $material = \App\Models\Material::find($linha['material_id']);

                \App\Models\ConsumoOp::create([
                    'op_id' => $op->id,
                    'material_id' => $material->id,
                    'quantidade' => $linha['quantidade'],
                    'unidade' => $material->unidade,
                    'categoria' => $material->categoria,
                ]);

                $material->decrement('estoque_atual', $linha['quantidade']);
                
                // Registrar movimentação de estoque
                \App\Models\MovimentacaoEstoque::create([
                    'material_id' => $material->id,
                    'tipo' => 'saida',
                    'quantidade' => $linha['quantidade'],
                    'motivo' => 'Consumo OP: ' . $request->numero_op,
                    'data' => now(),
                ]);
            }
        }

        return redirect()->route('ops.index')->with('success', 'OP criada e materiais abatidos com sucesso.');
    }

    public function show(Op $op)
    {
        return view('ops.show', compact('op'));
    }

    public function edit(Op $op)
    {
        $modelos = Modelo::all();
        return view('ops.edit', compact('op', 'modelos'));
    }

    public function update(Request $request, Op $op)
    {
        $request->validate([
            // Remover validação do numero_op para não permitir edição
            'data' => 'nullable|date',
            'descricao' => 'nullable',
            'quantidade' => 'required|integer',
            'modelo_id' => 'required|exists:modelos,id',
        ]);

        // Remover numero_op dos dados a serem atualizados
        $dados = $request->except(['numero_op']);
        $op->update($dados);

        return redirect()->route('ops.index')
            ->with('success', 'OP atualizada com sucesso.');
    }

    public function destroy(Op $op)
    {
        $op->delete();

        return redirect()->route('ops.index')
            ->with('success', 'OP excluída com sucesso.');
    }
}
